@extends('adminlte::page')

@section('title', 'Productos')

<head>
    <link rel="icon" href="{{ asset('icons/producto.png') }}" type="image/x-icon">
</head>

@section('content_header')
    <h1>Lista de Productos</h1>
    
@stop

@section('content')
<div class="d-flex justify-content-between align-items-center mb-3">
    <!-- Botón de agregar en la parte derecha -->
    <a href="{{ route('productos.create') }}" class="btn btn-primary shadow-lg" style="border-radius: 50px; padding: 10px 20px;">
        <i class="fas fa-plus"></i> Agregar
    </a>
</div>

<!-- Mensajes de éxito -->
@if (session('success'))
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <strong>¡Éxito!</strong> {{ session('success') }}
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
    <style>
        /* Estilos para el deslizador de notificaciones */

    </style>
@endif

<!-- Tabla de productos -->
<div class="card shadow-sm">
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-hover text-center">
                <thead class="bg-primary text-white">
                    <tr>
                        <th>ID</th>
                        <th>Nombre</th>
                        <th>Imagen</th>
                        <th>Precio</th>
                        <th>Stock</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach ($productos as $producto)
                        <tr>
                            <td>{{ $producto->id }}</td>
                            <td>{{ $producto->nombre }}</td>
                            <td>
                                <img src="{{ asset('storage/' . $producto->imagen) }}" alt="{{ $producto->nombre }}" style="width: 50px; height: 50px; object-fit: cover; border-radius: 10px;">
                            </td>
                            <td>S/ {{ number_format($producto->precio, 2) }}</td>
                            <td>
                                @if ($producto->stock < 5)
                                    <span class="badge bg-warning text-dark p-2 fw-bold">¡Stock bajo!</span>
                                @else
                                    {{ $producto->stock }}
                                @endif
                            </td>
                            <td class="d-flex justify-content-center gap-1">
                                <a href="{{ route('productos.edit', $producto->id) }}" class="btn btn-warning btn-sm rounded-pill shadow-sm">Editar</a>
                                <button type="button" class="btn btn-danger btn-sm rounded-pill shadow-sm delete-button" data-id="{{ $producto->id }}">Eliminar</button>
                                <form id="delete-form-{{ $producto->id }}" action="{{ route('productos.destroy', $producto->id) }}" method="POST" style="display: none;">
                                    @csrf
                                    @method('DELETE')
                                </form>
                            </td>
                        </tr>
                    @endforeach
                </tbody>
            </table>
        </div>

        <div class="d-flex justify-content-center mt-4">
            {{ $productos->links() }}
        </div>
    </div>
</div>

@stop

@section('footer')
    <div class="text-center">
    </div>
@stop

@section('js')
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script>
document.addEventListener('DOMContentLoaded', function () {
    const notificationIcon = document.getElementById('notificacion-stock-bajo');
    const notificationSlider = document.getElementById('notification-slider');

    if (notificationIcon && notificationSlider) {
        notificationIcon.addEventListener('mouseenter', function() {
            fetch('/api/productos/stock-bajo')
                .then(response => response.json())
                .then(data => {
                    const listElement = document.getElementById('productosStockBajoList');
                    listElement.innerHTML = '';  // Limpiamos cualquier contenido previo

                    data.forEach(producto => {
                        const listItem = document.createElement('li');
                        listItem.textContent = `${producto.nombre} - Stock: ${producto.stock}`;
                        listElement.appendChild(listItem);
                    });

                    notificationSlider.classList.add('open');
                })
                .catch(error => {
                    console.error('Error al cargar los productos:', error);
                });
        });

        document.addEventListener('click', function(event) {
            if (!notificationSlider.contains(event.target) && !notificationIcon.contains(event.target)) {
                notificationSlider.classList.remove('open');
            }
        });
    }
});

// Script para confirmación de eliminación con SweetAlert2
document.querySelectorAll('.delete-button').forEach(button => {
        button.addEventListener('click', function () {
            let id = this.getAttribute('data-id');
            Swal.fire({
                title: '¿Estás seguro?',
                text: "¡Esta acción no se puede deshacer!",
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#d33',
                cancelButtonColor: '#3085d6',
                confirmButtonText: 'Sí, eliminar',
                cancelButtonText: 'Cancelar'
            }).then((result) => {
                if (result.isConfirmed) {
                    document.getElementById('delete-form-' + id).submit();
                }
            });
        });
    });

    // Mostrar notificación si el stock es bajo (menos de 5)
    @foreach ($productos as $producto)
        @if ($producto->stock < 5)
            Swal.fire({
                title: '¡Atención!',
                text: 'El producto {{ $producto->nombre }} tiene bajo stock (menos de 5).',
                icon: 'warning',
                timer: 3000,
                showConfirmButton: false
            });
        @endif
    @endforeach
</script>
@endsection
