<?php
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Venta;
use App\Models\Producto;
use App\Models\Alquiler;

class VentaController extends Controller
{
    public function index()
    {
        $ventas = Venta::latest()->get();
        return view('ventas.ventatotal', compact('ventas'));
    }

    public function store(Request $request)
    {
        try {
            // Validar los datos recibidos
            $request->validate([
                'alquiler_id' => 'required|exists:alquileres,id', // Validar que el alquiler exista
                'productos' => 'required|array',
                'productos.*.id' => 'required|integer|exists:productos,id',
                'productos.*.cantidad' => 'required|integer|min:1',
                'monto_pagado' => 'required|numeric|min:0',
            ]);

            $totalVenta = 0;  // Variable para almacenar el total de la venta

            foreach ($request->productos as $producto) {
                $productoModel = Producto::find($producto['id']);

                if ($productoModel->stock < $producto['cantidad']) {
                    return response()->json([
                        'success' => false,
                        'error' => "Stock insuficiente para el producto: {$productoModel->nombre}"
                    ], 400);
                }

                // Calcular el subtotal de cada producto
                $subtotal = $producto['cantidad'] * $productoModel->precio;
                $totalVenta += $subtotal;  // Acumular el subtotal en el total de la venta

                // Crear el registro de la venta
                Venta::create([
                    'alquiler_id' => $request->alquiler_id, // Asociar con el alquiler
                    'producto_id' => $producto['id'],
                    'cantidad' => $producto['cantidad'],
                    'precio_total' => $subtotal,
                    'monto_pagado' => $request->monto_pagado,
                    'nombre_producto' => $productoModel->nombre,
                    'vuelto' => $request->monto_pagado - $totalVenta,  // Calcular el vuelto
                ]);

                // Actualizar el stock del producto
                $productoModel->stock -= $producto['cantidad'];
                $productoModel->save();
            }

            // Verificar si el monto pagado es suficiente para cubrir la venta
            if ($request->monto_pagado < $totalVenta) {
                return response()->json([
                    'success' => false,
                    'error' => "El monto pagado es insuficiente para la venta."
                ], 400);
            }

            return response()->json(['success' => true, 'message' => 'Venta registrada correctamente con alquiler.']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'error' => $e->getMessage()], 500);
        }
    }

    public function crear()
    {
        $alquileres = Alquiler::all(); // Asegúrate de tener el modelo Alquiler
        $productos = Producto::all();
        return view('ventas.crear', compact('alquileres', 'productos'));
    }

    public function destroy($id)
    {
        $venta = Venta::findOrFail($id);
        $venta->delete();

        return redirect()->route('ventas.index')->with('success', 'Venta eliminada correctamente.');
    }
}
