<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    /**
     * Mostrar la lista paginada de usuarios
     */
    public function index()
{
    $usuarios = User::orderBy('created_at', 'desc')->paginate(10);
    return view('usuarios.index', compact('usuarios'));
}

    /**
     * Mostrar formulario de creación de usuario
     */
    public function create()
    {
        return view('usuarios.create');
    }

    /**
     * Almacenar un nuevo usuario en la base de datos
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'foto' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'activo' => 'sometimes|boolean'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $fotoPath = null;
            if ($request->hasFile('foto')) {
                // Almacenar foto en el almacenamiento público
                $fotoPath = $request->file('foto')->store('profile-photos', 'public');
            }

            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'foto' => $fotoPath,
                'activo' => $request->activo ?? true
            ]);

            return redirect()->route('usuarios.index')
                ->with('success', 'Usuario creado exitosamente');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error al crear el usuario: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Mostrar formulario de edición de usuario
     */
    public function edit($id)
    {
        $usuario = User::findOrFail($id);
        return view('usuarios.edit', compact('usuario'));
    }

    /**
     * Actualizar un usuario existente
     */
    public function update(Request $request, $id)
    {
        $usuario = User::findOrFail($id);
    
        // Validación de los campos
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => [
                'required',
                'email',
                'max:255',
                Rule::unique('users')->ignore($usuario->id)
            ],
            'password' => 'nullable|string|min:8|confirmed',
            'foto' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'activo' => 'sometimes|boolean'
        ]);
    
        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
    
        try {
            // Preparando los datos para actualizar
            $data = [
                'name' => $request->name,
                'email' => $request->email,
                'activo' => $request->activo ?? $usuario->activo
            ];
    
            // Si se proporcionó una nueva contraseña, la actualizamos
            if ($request->filled('password')) {
                $data['password'] = Hash::make($request->password);
            }
    
            // Si se subió una nueva foto, la procesamos
            if ($request->hasFile('foto')) {
                // Eliminar la foto anterior si existe
                if ($usuario->foto && Storage::disk('public')->exists($usuario->foto)) {
                    Storage::disk('public')->delete($usuario->foto);
                }
    
                // Mover el archivo a la ubicación final dentro del almacenamiento público
                // Se guarda en la carpeta 'profile-photos' dentro de 'storage/app/public'
                $fotoPath = $request->file('foto')->store('profile-photos', 'public'); 
    
                // Guardamos la ruta de la foto en la base de datos
                $data['foto'] = $fotoPath;
            }
    
            // Actualizar los datos del usuario en la base de datos
            $usuario->update($data);
    
            // Redirigir al índice con un mensaje de éxito
            return redirect()->route('usuarios.index')
                ->with('success', 'Usuario actualizado exitosamente');
        } catch (\Exception $e) {
            // En caso de error, redirigimos de nuevo con el mensaje de error
            return redirect()->back()
                ->with('error', 'Error al actualizar el usuario: ' . $e->getMessage())
                ->withInput();
        }
    }
    

    
    
    

    /**
     * Eliminar un usuario
     */
    public function destroy($id)
{
    try {
        $usuario = User::findOrFail($id);

        // Eliminar foto si existe
        if ($usuario->foto && Storage::disk('public')->exists($usuario->foto)) {
            Storage::disk('public')->delete($usuario->foto);
        }

        $usuario->delete();

        // Redirigir con mensaje de éxito
        return redirect()->route('usuarios.index')->with('success', 'Usuario eliminado correctamente');
    } catch (\Exception $e) {
        // En caso de error, redirigir con mensaje de error
        return redirect()->route('usuarios.index')->with('error', 'Error al eliminar el usuario: ' . $e->getMessage());
    }
}

    /**
     * Cambiar estado activo/inactivo del usuario
     */
    public function cambiarEstado(Request $request, $id)
    {
        // Buscar al usuario por su ID
        $usuario = User::find($id);
        
        // Verificar si el usuario existe
        if (!$usuario) {
            return response()->json(['success' => false, 'message' => 'Usuario no encontrado.']);
        }
    
        // Actualizar el estado del usuario
        $usuario->activo = $request->activo; // Cambiar el estado de activo/inactivo
        $usuario->save(); // Guardar los cambios
        
        // Retornar una respuesta JSON
        return response()->json(['success' => true, 'activo' => $usuario->activo]);
    }


    public function esSuperAdmin($id)
{
    $user = User::findOrFail($id);
    return response()->json([
        'es_super_admin' => $user->hasRole('super-admin')
    ]);
}


}
