<?php

namespace App\Http\Controllers;

use App\Models\Pelicula;
use App\Models\Categoria;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Storage;
use Rap2hpoutre\FastExcel\FastExcel;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests; // Trae el trait adecuado
use App\Http\Controllers\Controller;

class PeliculaController extends Controller
{
  

    public function index(Request $request) // Añadí el parámetro Request
{
    // Si se pasa un filtro por categoría, aplica el filtro
    $query = Pelicula::with('categoria'); // Asegúrate de que tienes la relación 'categoria' en tu modelo Pelicula

    if ($request->has('categoria_id') && $request->categoria_id != '') {
        $query->where('categoria_id', $request->categoria_id); // Filtramos las películas por categoría
    }

    // Obtenemos el término de búsqueda
    $search = $request->input('search');
    
    // Si hay un término de búsqueda, lo aplicamos a la consulta
    if ($search) {
        $query->where('titulo', 'like', '%' . $search . '%'); // Filtramos por título de la película
    }

    // Obtenemos las películas paginadas (10 por página por defecto)
    $peliculas = $query->paginate($request->get('entries', 12));  // Usamos 'entries' para determinar cuántos resultados mostrar

    // Obtener todas las categorías para el filtro del formulario
    $categorias = Categoria::all(); // Asegúrate de que el modelo Categoria existe y está bien relacionado

    // Pasar las películas y las categorías a la vista
    return view('peliculas.index', compact('peliculas', 'categorias'));
}


    public function create()
{
    // Obtener todas las categorías de la base de datos
    $categorias = Categoria::all(); // Asegúrate de que Categoria es el nombre de tu modelo

    // Retornar la vista y pasar las categorías
    return view('peliculas.create', compact('categorias'));
}

    public function store(Request $request)
    {
        $request->validate([
            'imagen' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'titulo' => 'required|string|max:255',
            'horas' => 'nullable|integer|min:0',
            'minutos' => 'nullable|integer|min:0|max:59',
            'formato' => 'required|string|max:50',
            'categoria_id' => 'required|exists:categorias,id',
            'estado' => 'required|in:disponible,alquilada',
        ]);

        $duracion = ($request->horas * 60) + $request->minutos;
        $imagenPath = $request->hasFile('imagen') ? $request->file('imagen')->store('peliculas', 'public') : null;

        Pelicula::create([
            'titulo' => $request->titulo,
            'duracion' => $duracion,
            'formato' => $request->formato,
            'categoria_id' => $request->categoria_id,
            'estado' => $request->estado,
           
        ]);

        return redirect()->route('peliculas.index')->with('success', 'Película guardada correctamente.');
    }

    public function show(Pelicula $pelicula)
    {
        return view('peliculas.show', compact('pelicula'));
    }

    public function edit(Pelicula $pelicula)
    {
        $categorias = Categoria::all();
        return view('peliculas.edit', compact('pelicula', 'categorias'));
    }

    public function update(Request $request, Pelicula $pelicula)
    {
        $request->validate([
            'titulo' => 'required|string|max:255',
            'duracion' => 'required|integer',
            'formato' => 'required|string|max:255',
            'categoria_id' => 'required|exists:categorias,id',
            'estado' => 'required|in:disponible,alquilada',
            'imagen' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048'
        ]);

        if ($request->hasFile('imagen')) {
            if ($pelicula->imagen) {
                Storage::disk('public')->delete($pelicula->imagen);
            }
            $imagenPath = $request->file('imagen')->store('peliculas', 'public');
            $pelicula->imagen = $imagenPath;
        }

        $pelicula->update($request->except('imagen'));

        return redirect()->route('peliculas.index')->with('success', 'Película actualizada correctamente.');
    }

    public function destroy(Pelicula $pelicula)
    {
        if ($pelicula->imagen) {
            Storage::disk('public')->delete($pelicula->imagen);
        }
        $pelicula->delete();
        return redirect()->route('peliculas.index')->with('success', 'Película eliminada correctamente.');
    }

    public function search(Request $request)
    {
        $term = $request->get('term');
        $categoriaId = $request->get('categoria_id');
        
        $query = Pelicula::query();
    
        if ($term) {
            $query->where('titulo', 'like', "%$term%");
        }
    
        if ($categoriaId) {
            $query->where('categoria_id', $categoriaId);
        }
    
        $peliculas = $query->limit(10)->get(['id', 'titulo', 'duracion']); // Añade 'duracion'
    
        return response()->json([
            'peliculas' => $peliculas
        ]);
    }


    public function exportToPDF()
    {
        $peliculas = Pelicula::with('categoria')->get();
        $pdf = Pdf::loadView('peliculas.pdf', compact('peliculas'));
        return $pdf->download('peliculas.pdf');
    }

    public function import(Request $request)
    {
        $file = $request->file('file');
    
        (new FastExcel)->import($file, function ($row) {
            return Pelicula::create([
                'titulo'       => $row['titulo'],  // Verifica que coincida con el encabezado
                'duracion'     => $row['duracion'],
                'formato'      => $row['formato'],
                'categoria_id' => isset($row['categoria_id']) ? $row['categoria_id'] : null,
                'estado'       => $row['estado'],
            ]);
        });
    
        return back()->with('success', 'Películas importadas correctamente.');
    }

    public function exportExcel()
    {
        $peliculas = Pelicula::all(['id', 'titulo', 'duracion', 'formato', 'categoria_id', 'estado']);

        return (new FastExcel($peliculas))->download('peliculas.xlsx');
    }
}
