<?php

namespace App\Http\Controllers;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use App\Models\Cobro;
use App\Models\Pelicula;
use App\Models\Alquiler;

class CobroController extends Controller
{
    /**
     * Muestra la lista de cobros con paginación, incluyendo el precio de la película y el vuelto.
     */
    public function index()
{
    // Obtener todos los cobros con su relación alquiler
    $cobros = Cobro::with('alquiler.pelicula')->latest()->paginate(10);

    // Calcular el total cobrado
    $totalCobrado = $cobros->sum('monto');

    // Calcular el monto promedio de los cobros
    $montoPromedio = $cobros->avg('monto');

    // Calcular los cobros pendientes (ejemplo: aquellos con vuelto negativo)
    $cobrosPendientes = $cobros->where('vuelto', '<', 0)->count();

    // Calcular los cobros completados (aquellos con vuelto positivo o cero)
    $cobrosCompletados = $cobros->where('vuelto', '>=', 0)->count();

    // Obtener los totales por película
    $totalesPorPelicula = Alquiler::join('peliculas', 'alquileres.pelicula_id', '=', 'peliculas.id')
        ->selectRaw('peliculas.titulo AS nombre_pelicula, SUM(alquileres.precio) AS total_precio')
        ->groupBy('peliculas.titulo')
        ->pluck('total_precio', 'nombre_pelicula');

    // Pasar las variables necesarias a la vista
    return view('cobros.index', compact('cobros', 'totalesPorPelicula', 'totalCobrado', 'montoPromedio', 'cobrosPendientes', 'cobrosCompletados'));
}


    /**
     * Guarda un nuevo cobro en la base de datos, calculando el vuelto.
     */
    public function store(Request $request)
    {
        $request->validate([
            'alquiler_id' => 'required|exists:alquileres,id',
            'monto' => 'required|numeric|min:0',
        ]);

        // Buscar el alquiler correspondiente
        $alquiler = Alquiler::findOrFail($request->alquiler_id);

        // Obtener el precio de la película
        $precioPelicula = $alquiler->precio;

        // Calcular el vuelto
        $vuelto = $request->monto - $precioPelicula;

        // Registrar el cobro
        Cobro::create([
            'alquiler_id' => $alquiler->id,
            'monto' => $request->monto,
            'vuelto' => $vuelto,
            'fecha_cobro' => now(),
        ]);

        // Actualizar el estado del alquiler a "pagado"
        $alquiler->estado = 'pagado';
        $alquiler->save();

        return back()->with('success', 'Cobro registrado correctamente.');
    }
    


    
}