<?php

namespace App\Http\Controllers;
use Carbon\Carbon;
use Illuminate\Http\Request;
use App\Models\Alquiler;
use App\Models\Pelicula;
use App\Models\Sala;
use App\Models\Cliente;
use App\Models\Cobro;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Log;
use Barryvdh\DomPDF\Facade\Pdf;
use Rap2hpoutre\FastExcel\FastExcel;

class AlquilerController extends Controller
{
    public function index()
    {
        $alquileres = Alquiler::with(['pelicula', 'sala', 'cliente'])->paginate(10);
        return view('alquileres.index', compact('alquileres'));
    }

    public function create()
    {
        $peliculas = Pelicula::where('estado', 'disponible')->get();
        $salas = Sala::where('estado', 'disponible')->get(); // Solo salas disponibles
        $clientes = Cliente::all();

        return view('alquileres.create', compact('peliculas', 'salas', 'clientes'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'pelicula_id' => 'required|exists:peliculas,id',
            'sala_id' => 'required|exists:salas,id',
            'cliente' => 'required|string|max:255',
            'hora_entrada' => 'required|date_format:H:i',
            'hora_salida' => 'required|date_format:H:i|after:hora_entrada',
            'precio' => 'required|numeric|min:0',
        ]);

        $pelicula = Pelicula::find($request->pelicula_id);
        $sala = Sala::find($request->sala_id);

        if ($pelicula->estado !== 'disponible') {
            return redirect()->route('alquileres.create')
                ->withErrors(['error' => 'Esta película no está disponible para alquiler.']);
        }

        if ($sala->estado !== 'disponible') {
            return redirect()->route('alquileres.create')
                ->withErrors(['error' => 'Esta sala ya está ocupada.']);
        }
// Convertir la fecha a la zona horaria de Lima
$fecha_actual = Carbon::now('America/Lima')->toDateTimeString();
        // Registrar alquiler
        $alquiler = Alquiler::create($request->only([
            'pelicula_id', 'sala_id', 'cliente', 'hora_entrada', 'hora_salida', 'precio'
        ]));

        // Cambiar estado de la película y la sala
        $pelicula->estado = 'alquilada';
        $pelicula->save();

        $sala->estado = 'ocupado';
        $sala->save();

        return redirect()->route('alquileres.index')->with('success', 'Alquiler registrado correctamente.');
    }

    public function show($id)
    {
        
        $alquiler = Alquiler::findOrFail($id);
         // Convertir fecha a la zona horaria correcta antes de mostrarla
    $alquiler->created_at = Carbon::parse($alquiler->created_at)->timezone('America/Lima');
        return view('alquileres.show', compact('alquiler'));
    }

    public function edit($id)
    {
        $alquiler = Alquiler::findOrFail($id);
        $peliculas = Pelicula::where('estado', 'disponible')->orWhere('id', $alquiler->pelicula_id)->get();
        $salas = Sala::all();
        $clientes = Cliente::all();

        return view('alquileres.edit', compact('alquiler', 'peliculas', 'salas', 'clientes'));
    }

    public function update(Request $request, $id)
    {
        $alquiler = Alquiler::findOrFail($id);

        $request->validate([
            'pelicula_id' => 'required|exists:peliculas,id',
            'cliente' => 'required|string|max:255',
            'hora_entrada' => 'required|date_format:H:i',
            'hora_salida' => 'required|date_format:H:i|after:hora_entrada',
            'precio' => 'required|numeric|min:0',
        ]);

        Log::info('Datos recibidos para actualización:', $request->all());

        $data = $request->only(['pelicula_id', 'cliente', 'hora_entrada', 'hora_salida', 'precio']);

        $alquiler->update($data);

        return redirect()->route('alquileres.index')->with('success', 'Alquiler actualizado correctamente.');
    }

    public function destroy($id)
    {
        try {
            $alquiler = Alquiler::findOrFail($id);

            if ($alquiler->pelicula) {
                Pelicula::where('id', $alquiler->pelicula_id)->update(['estado' => 'disponible']);
            }

            if ($alquiler->sala) {
                Sala::where('id', $alquiler->sala_id)->update(['estado' => 'disponible']);
            }

            $alquiler->delete();

            return redirect()->route('alquileres.index')->with('success', 'Alquiler eliminado correctamente.');
        } catch (ModelNotFoundException $e) {
            return redirect()->route('alquileres.index')->with('error', 'Alquiler no encontrado.');
        }
    }

    public function cobrar(Request $request, $id)
{
    Log::info('Datos recibidos:', $request->all());

    $metodo_pago = $request->input('metodo_pago');
    $monto = $request->input('monto', null);

    // Si el método de pago es efectivo, validar el monto
    if ($metodo_pago === 'efectivo') {
        if (is_null($monto) || $monto === '') {
            return response()->json(['mensaje' => 'Monto no recibido'], 400);
        }

        $alquiler = Alquiler::with(['pelicula', 'sala'])->findOrFail($id);

        if ($monto < $alquiler->precio) {
            return response()->json(['mensaje' => 'Monto insuficiente'], 400);
        }
    } else {
        // Si es Yape, obtener el alquiler sin validar monto
        $alquiler = Alquiler::with(['pelicula', 'sala'])->findOrFail($id);
        $monto = $alquiler->precio; // Se asume que se pagó el total
    }

    // Registrar el cobro
    $cobro = Cobro::create([
        'alquiler_id' => $alquiler->id,
        'monto' => $monto,
        'fecha_cobro' => Carbon::now('America/Lima')->toDateTimeString(),
    ]);

    // Actualizar el estado del alquiler a "pagado"
    $alquiler->estado = 'pagado';
    
    $alquiler->save();

    // Restaurar estado de la película y sala
    if ($alquiler->pelicula) {
        $alquiler->pelicula->estado = 'disponible';
        $alquiler->pelicula->save();
    }

    if ($alquiler->sala) {
        $alquiler->sala->estado = 'disponible';
        $alquiler->sala->save();
    }

    return response()->json([
        'mensaje' => 'Pago realizado con éxito',
        'cobro' => $cobro,
    ]);
}


    public function generarTicket($id)
    {
        $alquiler = Alquiler::findOrFail($id);

        $pdf = Pdf::loadView('alquileres.ticket', compact('alquiler'))
            ->setPaper([0, 0, 226.500, 600], 'portrait');

        return $pdf->stream("ticket_alquiler_{$alquiler->id}.pdf");
    }

    public function exportExcel()
{
    // Obtener los datos de los alquileres con la relación a la película
    $alquileres = Alquiler::with('pelicula') // Cargar la relación 'pelicula' de cada alquiler
        ->get(['id', 'pelicula_id', 'cliente', 'sala_id', 'hora_entrada', 'hora_salida', 'precio']);

    // Usar FastExcel para exportar los datos
    $data = $alquileres->map(function ($alquiler) {
        return [
            'ID' => $alquiler->id,
            'Pelicula' => $alquiler->pelicula ? $alquiler->pelicula->titulo : 'N/A', // Obtener el título de la película (campo 'titulo')
            'Cliente' => $alquiler->cliente,
            'Sala ID' => $alquiler->sala_id,
            'Hora Entrada' => $alquiler->hora_entrada,
            'Hora Salida' => $alquiler->hora_salida,
            'Precio' => $alquiler->precio,
        ];
    });

    // Exportar con FastExcel
    return (new FastExcel($data))->download('alquileres.xlsx');
}
    
}
